/////////////////////////////////////////////////////////////////////////////
//
//	Linked List demo
//
//	Author: Chad Knudson
//
//	History:
//
//		05/01/97	Created. [ChadK]
//
/////////////////////////////////////////////////////////////////////////////
#include <stdio.h>
#include <stdlib.h>

typedef struct _node 
{
	int iValue;
	struct _node * pNodeNext;
} NODE;

typedef struct _linkedlist
{
	NODE *	pNodeHead;
	int		cElements;
} LinkedList;

// Add bool type if needed
#ifndef BOOL
#define BOOL	int
#define TRUE	1
#define FALSE	0
#endif

/////////////////////////////////////////////////////////////////////////////
//							Function Prototypes
/////////////////////////////////////////////////////////////////////////////
BOOL FAddLinkedListElement(LinkedList *, int);
void PrintListElements(LinkedList *);
void RemoveAllElementsFromList(LinkedList *);

/////////////////////////////////////////////////////////////////////////////
//								Main Program
/////////////////////////////////////////////////////////////////////////////
int main (int argc, char ** argv) 
{
	LinkedList List;
	int iTestValuesArray[5] = { 10, 20, 30, 5, 40 };
	int iTestValuesIndex;
	
	/////////////////////////////////////////////////////////////////////////
	// Initialize List member fields
	/////////////////////////////////////////////////////////////////////////
	List.pNodeHead = NULL;
	List.cElements = 0;

	/////////////////////////////////////////////////////////////////////////
	// Add some values to the list
	/////////////////////////////////////////////////////////////////////////
	for (iTestValuesIndex = 0; iTestValuesIndex < 5; iTestValuesIndex++)
	{
		if (!FAddLinkedListElement(&List, iTestValuesArray[iTestValuesIndex]))
		{
			fprintf(stderr, "Error adding a node to the linked list\n");
			exit(1);
		}
	}

	/////////////////////////////////////////////////////////////////////////
	// Print out the values in the list
	/////////////////////////////////////////////////////////////////////////
	PrintListElements(&List);

	/////////////////////////////////////////////////////////////////////////
	// Do our housekeeping -- free up all of the dynamically allocated memory
	/////////////////////////////////////////////////////////////////////////
	RemoveAllElementsFromList(&List);
 	
	return(0);
}

/////////////////////////////////////////////////////////////////////////////
//
//	FAddLinkedListElement
//
//	Purpose:	Add an element to the list with an integer value.
//
//	Method:		Adds a node to the end of the list.
//
//	Returns:	TRUE  on success,
//				FALSE on failure.
//
//	Notes:
//
//	Owner:		ChadK
//
//	History:
//
//		05/01/97:	Created. [ChadK]
//	
/////////////////////////////////////////////////////////////////////////////
BOOL FAddLinkedListElement(LinkedList * pList , int iValue)
{
	NODE * pNodeCurrent;
	NODE * pNodeNew;
	NODE * pNodePrevious;
	
	/////////////////////////////////////////////////////////////////////////
	// Create new node
	/////////////////////////////////////////////////////////////////////////
	pNodeNew = (NODE *)malloc(sizeof(NODE));

	if (pNodeNew == NULL)
	   return(FALSE);

	/////////////////////////////////////////////////////////////////////////
	// Initialize the new node's pNodeNext to NULL and place iValue in the
	// node
	/////////////////////////////////////////////////////////////////////////
	pNodeNew->pNodeNext = NULL;
	pNodeNew->iValue	= iValue;
	
	/////////////////////////////////////////////////////////////////////////
	// Get things ready to traverse the list
	/////////////////////////////////////////////////////////////////////////
	pNodePrevious		= NULL;
	pNodeCurrent		= pList->pNodeHead;
	
	/////////////////////////////////////////////////////////////////////////
	// Traverse the list -- we want the new node to be at the end of the list
	/////////////////////////////////////////////////////////////////////////
	while (pNodeCurrent != NULL)
	{
		pNodePrevious	= pNodeCurrent;
		pNodeCurrent	= pNodeCurrent->pNodeNext;
	}

	/////////////////////////////////////////////////////////////////////////
	// When we get here, pNodePrevious points to the node right before the
	// place where the new node should go.
	/////////////////////////////////////////////////////////////////////////
	if (pNodePrevious != NULL)
	{
		/////////////////////////////////////////////////////////////////////
		// Fixup the links with the previous node and the new node
		/////////////////////////////////////////////////////////////////////
		pNodeNew->pNodeNext = pNodePrevious->pNodeNext;
		pNodePrevious->pNodeNext = pNodeNew;
	}
	else
	{
		/////////////////////////////////////////////////////////////////////
		// This must be the first node of the list
		/////////////////////////////////////////////////////////////////////
		pList->pNodeHead = pNodeNew;

		/////////////////////////////////////////////////////////////////////
		// pNodeNew->pNodeNext was already initialized to NULL so we don't 
		// have to do anything with it here
		/////////////////////////////////////////////////////////////////////
	}

	pList->cElements++;	// Increase our count of elements in the list

	return(TRUE);
	
}

/////////////////////////////////////////////////////////////////////////////
//
//	PrintListElements
//
//	Purpose:	Print the values in the nodes of an integer list.
//
//	Method:		
//
//	Returns:	Nothing.
//
//	Notes:
//
//	Owner:		ChadK
//
//	History:
//
//		05/01/97:	Created. [ChadK]
//	
/////////////////////////////////////////////////////////////////////////////
void PrintListElements(LinkedList * pList)
{
	NODE *	pNodeCurrent;
	int		iElement = 0;

	pNodeCurrent = pList->pNodeHead;
	
	printf("The linked list has %d nodes.\n\n", pList->cElements);

	/////////////////////////////////////////////////////////////////////////
	// Traverse the list -- we want the new node to be at the end of the list
	/////////////////////////////////////////////////////////////////////////
	while (pNodeCurrent != NULL)
	{
		printf("%d", pNodeCurrent->iValue);
		
		if (++iElement < pList->cElements)
		{
			printf(", ");
		}
		else
		{
			printf(".\n");
		}

		pNodeCurrent = pNodeCurrent->pNodeNext;
	}
}

/////////////////////////////////////////////////////////////////////////////
//
//	PrintListElements
//
//	Purpose:	Print the values in the nodes of an integer list.
//
//	Method:		
//
//	Returns:	Nothing.
//
//	Notes:
//
//	Owner:		ChadK
//
//	History:
//
//		05/01/97:	Created. [ChadK]
//	
/////////////////////////////////////////////////////////////////////////////
void RemoveAllElementsFromList(LinkedList * pList)
{
	NODE *	pNodeKill;
	NODE *	pNodeCurrent;

	pNodeKill	 = NULL;
	pNodeCurrent = pList->pNodeHead;
	
	/////////////////////////////////////////////////////////////////////////
	// Traverse the list -- we want to free each node's memory
	/////////////////////////////////////////////////////////////////////////
	while (pNodeCurrent != NULL)
	{
		pNodeKill		= pNodeCurrent;
		pNodeCurrent	= pNodeCurrent->pNodeNext;
		free(pNodeKill);
	}

	/////////////////////////////////////////////////////////////////////////
	// Re-initialize List member fields
	/////////////////////////////////////////////////////////////////////////
	pList->pNodeHead = NULL;
	pList->cElements = 0;

}
